unit uSingletonTest;

interface

uses
  TestFramework
  , uTestCaseException          // for tests that test for exceptions
  , uSimpleSingleton
  , uSimpleSingleton2
  , uSimpleSingleton3
  , uSimpleSingleton4
  , uInheritableSingleton
  ;

type

  TSimpleSingletonTest = class(TTestCase)
  private

  published
    procedure TestNotNil;
    procedure TestCreateError;
    procedure TestFreeError;
  end;

  TSimpleSingleton2Test = class(TTestCase)
  private

  published
    procedure TestNotNil;
    procedure TestCreateError;
    procedure TestFreeError;
  end;

  TSimpleSingleton3Test = class(TTestCase)
  private

  published
    procedure TestNotNil;
    procedure TestCreateError;
    procedure TestFreeError;
    procedure TestCreateDescendant;
    procedure TestCreateDescendants;
  end;

  TSimpleSingleton4Test = class(TTestCaseException)
  private

  published
    procedure TestNotNil;
    procedure TestCreateError;
    procedure TestFreeError;
    procedure TestCreateDescendant;
    procedure TestCreateDescendants;
  end;

  TInheritableSingletonTest = class(TTestCase)
  private

  published
    procedure TestNotNil;
    procedure TestCreateError;
    procedure TestFreeError;
    procedure TestCreateDescendant;
    procedure TestCreateDescendants;
    procedure TestOverriddenCreate;
    procedure TestOverriddenInitialise;
  end;

implementation

uses
  sysutils                    // for Exception class
  ;

{ TSingletonTest }

procedure TSimpleSingletonTest.TestCreateError;
var
  ssTmp: TSimpleSingleton;
begin
  // See what happens when we call the Create method
  ssTmp := nil;
  ssTmp := TSimpleSingleton.Create;
  // Ideally, we don't want anything to happen
  Check(not Assigned(ssTmp), 'SimpleSingleton is assigned');
end;

procedure TSimpleSingletonTest.TestFreeError;
var
  ssTmp: TSimpleSingleton;
  nTmp: integer;
begin
// Uncomment this code with care - it thoroughly messes things up!
{  // See what happens when we call the singleton's free method
  ssTmp := nil;
  // Get a reference to the singleton
  ssTmp := SimpleSingleton;
  Check(Assigned(ssTmp), 'SimpleSingleton not assigned (1st time)');
  // Now call the free method
  ssTmp.Free;
  ssTmp := nil;
  // Now try getting another reference
  ssTmp := SimpleSingleton;
  // If we get this far, make sure we've got a valid reference
  Check(Assigned(ssTmp), 'SimpleSingleton not assigned (2nd time)');
  // Try to reference the singleton's property
  ssTmp.Value := 100;
  // Try to call the object's Create method
  ssTmp.Create;}
end;

procedure TSimpleSingletonTest.TestNotNil;
var
  ssTmp: TSimpleSingleton;
begin
  // We should be able to get the singleton from the function
  ssTmp := nil;
  ssTmp := SimpleSingleton;
  Check(Assigned(ssTmp), 'SimpleSingleton not assigned');
end;

{ TSimpleSingleton2Test }

procedure TSimpleSingleton2Test.TestCreateError;
var
  ssTmp: TSimpleSingleton2;
begin
  // See what happens when we call the Create method
  ssTmp := nil;
  ssTmp := TSimpleSingleton2.Create;
  // Ideally, we don't want anything to happen
  Check(not Assigned(ssTmp), 'SimpleSingleton2 is assigned');
end;

procedure TSimpleSingleton2Test.TestFreeError;
var
  ssTmp: TSimpleSingleton2;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  ssTmp := nil;
  // Get a reference to the singleton
  ssTmp := SimpleSingleton2;
  Check(Assigned(ssTmp), 'SimpleSingleton2 not assigned (1st time)');
  // Now call the free method
  ssTmp.Free;
  ssTmp := nil;
  // Now try getting another reference
  ssTmp := SimpleSingleton2;
  // If we get this far, make sure we've got a valid reference
  Check(Assigned(ssTmp), 'SimpleSingleton2 not assigned (2nd time)');
  // Try to reference the singleton's property
  ssTmp.Value := 100;
  // Try to call the object's Create method
  ssTmp.Create;
end;

procedure TSimpleSingleton2Test.TestNotNil;
var
  ssTmp: TSimpleSingleton2;
begin
  // We should be able to get the singleton from the function
  ssTmp := nil;
  ssTmp := SimpleSingleton2;
  Check(Assigned(ssTmp), 'SimpleSingleton2 not assigned');
end;

{ TSimpleSingleton3Test }

{ Create two descendant classes }

type
  TSimpleSingleton3Descendant1 = class(TSimpleSingleton3)
  private
    FMyValue1: integer;
  public
    property MyValue1: integer
      read FMyValue1
      write FMyValue1;
  end;

  TSimpleSingleton3Descendant2 = class(TSimpleSingleton3)
  private
    FMyValue2: integer;
  public
    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

procedure TSimpleSingleton3Test.TestCreateError;
var
  ssTmp, ssTmp2: TSimpleSingleton3;
begin
  // See what happens when we call the Create method
  ssTmp := nil;
  ssTmp2 := nil;
  ssTmp := TSimpleSingleton3.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  ssTmp2 := TSimpleSingleton3.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
end;

procedure TSimpleSingleton3Test.TestFreeError;
var
  ssTmp, ssTmp2: TSimpleSingleton3;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  ssTmp := nil;
  // Get references to the singleton
  ssTmp := TSimpleSingleton3.Create;
  ssTmp2 := TSimpleSingleton3.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
  // Now call the free method. Should get an EAbort exception
  try
    ssTmp.Free;
    fail('EAbort exception not raised');
  except
    on E: Exception do
    begin
      if E.ClassType <> EAbort then
        raise;
    end
  end;
  ssTmp := nil;
  // Try to reference the singleton's property
  ssTmp2.Value := 100;
  // Get another reference to the singleton
  ssTmp := TSimpleSingleton3.Create;
  // Check that we've still got the value
  Check(ssTmp.Value = 100, 'Value not retained');
end;

procedure TSimpleSingleton3Test.TestNotNil;
var
  ssTmp: TSimpleSingleton3;
begin
  // We should be able to get the singleton from the function
  ssTmp := nil;
  ssTmp := TSimpleSingleton3.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
end;

procedure TSimpleSingleton3Test.TestCreateDescendant;
var
  ss3Tmp: TSimpleSingleton3;
  ss3d1Tmp, ss3d1Tmp2: TSimpleSingleton3Descendant1;
begin
  // See what happens when we create a descendant of TSimpleSingleton3
  // First, make sure the singleton doesn't exist
  ss3Tmp := TSimpleSingleton3.Create;         // Get a reference to it
  if Assigned(ss3Tmp) then
    ss3Tmp.DestroyInstance;                     // Destroy it
  // Now do the test. Create two instances of the descendant type, and check
  // that both variables point to the same instance
  ss3d1Tmp := nil;
  ss3d1Tmp2 := nil;
  ss3d1Tmp := TSimpleSingleton3Descendant1.Create;
  Check(ss3d1Tmp.InitCount = 1, 'InitCount is not 1');
  ss3d1Tmp2 := TSimpleSingleton3Descendant1.Create;
  Check(ss3d1Tmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(ss3d1Tmp), 'ss3d1Tmp not assigned');
  Check(Assigned(ss3d1Tmp2), 'ss3d1Tmp2 not assigned');
  Check(ss3d1Tmp = ss3d1Tmp2, 'Singleton references are not the same');
end;

procedure TSimpleSingleton3Test.TestCreateDescendants;
var
  ss3Tmp: TSimpleSingleton3;
  ss3d1Tmp: TSimpleSingleton3Descendant1;
  ss3d2Tmp: TSimpleSingleton3Descendant2;
begin
  // See what happens when we create descendants of more than one type
  // First, make sure the singleton doesn't exist
{  ss3Tmp := TSimpleSingleton3.Create;         // Get a reference to it
  if Assigned(ss3Tmp) then
    ss3Tmp.DestroyInstance;                     // Destroy it}
  // Now do the test
  // Create a descendant of the first type and check its class name
  ss3d1Tmp := nil;
  ss3d1Tmp := TSimpleSingleton3Descendant1.Create;
  Check(ss3d1Tmp.ClassName = 'TSimpleSingleton3Descendant1',
                             'ss3d1Tmp is the wrong class (' +
                             ss3d1Tmp.ClassName + ')');
  // Create a descendant of the second type and check its class name
  ss3d2Tmp := nil;
  ss3d2Tmp := TSimpleSingleton3Descendant2.Create;
  Check(ss3d2Tmp.ClassName = 'TSimpleSingleton3Descendant2',
                             'ss3d2Tmp is the wrong class (' +
                             ss3d2Tmp.ClassName + ')');
end;

{ TSimpleSingleton4Test }

{ Create two descendant classes }

type
  TSimpleSingleton4Descendant1 = class(TSimpleSingleton4)
  private
    FMyValue1: integer;
  public
    property MyValue1: integer
      read FMyValue1
      write FMyValue1;
  end;

  TSimpleSingleton4Descendant2 = class(TSimpleSingleton4)
  private
    FMyValue2: integer;
  public
    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

procedure TSimpleSingleton4Test.TestCreateError;
var
  ssTmp, ssTmp2: TSimpleSingleton4;
begin
  // See what happens when we call the Create method
  ssTmp := nil;
  ssTmp2 := nil;
  ssTmp := TSimpleSingleton4.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  ssTmp2 := TSimpleSingleton4.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
end;

procedure TSimpleSingleton4Test.TestFreeError;
var
  ssTmp, ssTmp2: TSimpleSingleton4;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  ssTmp := nil;
  // Get references to the singleton
  ssTmp := TSimpleSingleton4.Create;
  ssTmp2 := TSimpleSingleton4.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
  // Now call the free method. Should get an EAbort exception
  try
    ssTmp.Free;
    fail('EAbort exception not raised');
  except
    on E: Exception do
    begin
      if E.ClassType <> EAbort then
        raise;
    end
  end;
  ssTmp := nil;
  // Try to reference the singleton's property
  ssTmp2.Value := 100;
  // Get another reference to the singleton
  ssTmp := TSimpleSingleton4.Create;
  // Check that we've still got the value
  Check(ssTmp.Value = 100, 'Value not retained');
end;

procedure TSimpleSingleton4Test.TestNotNil;
var
  ssTmp: TSimpleSingleton4;
begin
  // We should be able to get the singleton from the function
  ssTmp := nil;
  ssTmp := TSimpleSingleton4.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
end;

procedure TSimpleSingleton4Test.TestCreateDescendant;
var
  ss4Tmp: TSimpleSingleton4;
  ss4d1Tmp, ss4d1Tmp2: TSimpleSingleton4Descendant1;
begin
  // See what happens when we create a descendant of TSimpleSingleton4
  // First, make sure the singleton doesn't exist
  ss4Tmp := TSimpleSingleton4.Create;         // Get a reference to it
  if Assigned(ss4Tmp) then
    ss4Tmp.DestroyInstance;                     // Destroy it
  // Now do the test. Create two instances of the descendant type, and check
  // that both variables point to the same instance
  ss4d1Tmp := nil;
  ss4d1Tmp2 := nil;
  ss4d1Tmp := TSimpleSingleton4Descendant1.Create;
  Check(ss4d1Tmp.InitCount = 1, 'InitCount is not 1');
  ss4d1Tmp2 := TSimpleSingleton4Descendant1.Create;
  Check(ss4d1Tmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(ss4d1Tmp), 'ss4d1Tmp not assigned');
  Check(Assigned(ss4d1Tmp2), 'ss4d1Tmp2 not assigned');
  Check(ss4d1Tmp = ss4d1Tmp2, 'Singleton references are not the same');
end;

procedure TSimpleSingleton4Test.TestCreateDescendants;
var
  ss4Tmp: TSimpleSingleton4;
  ss4d1Tmp: TSimpleSingleton4Descendant1;
  ss4d2Tmp: TSimpleSingleton4Descendant2;
begin
  // See what happens when we create descendants of more than one type
  // We should get an ESimpleSingleton4 exception - if we get one, the test will
  // pass.
  Self.SetExpectedException(ESimpleSingleton4);
  // First, make sure the singleton doesn't exist
  ss4Tmp := TSimpleSingleton4.Create;         // Get a reference to it
  if Assigned(ss4Tmp) then
    ss4Tmp.DestroyInstance;                     // Destroy it
  // Now do the test
  // Create a descendant of the first type and check its class name
  ss4d1Tmp := nil;
  ss4d1Tmp := TSimpleSingleton4Descendant1.Create;
  Check(ss4d1Tmp.ClassName = 'TSimpleSingleton4Descendant1',
                             'ss4d1Tmp is the wrong class (' +
                             ss4d1Tmp.ClassName + ')');
  // Create a descendant of the second type and check its class name
  ss4d2Tmp := nil;
  // The next line should throw an ESimpleSingleton4 exception
  ss4d2Tmp := TSimpleSingleton4Descendant2.Create;
  Check(ss4d2Tmp.ClassName = 'TSimpleSingleton4Descendant2',
                             'ss4d2Tmp is the wrong class (' +
                             ss4d2Tmp.ClassName + ')');
end;

{ TInheritableSingletonTest }

{ Create two descendant classes }

type
  TInheritableSingletonDescendant1 = class(TInheritableSingleton)
  private
    FMyValue1: integer;
  public
    property MyValue1: integer
      read FMyValue1
      write FMyValue1;
  end;

  TInheritableSingletonDescendant2 = class(TInheritableSingleton)
  private
    FMyValue2: integer;
  public
    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

  TInheritableSingletonDescendant3 = class(TInheritableSingleton)
  private
    FMyValue2: integer;
  protected
    procedure Initialise; override;
  public

    constructor Create; override;

    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

procedure TInheritableSingletonTest.TestCreateError;
var
  ssTmp, ssTmp2: TInheritableSingleton;
begin
  // See what happens when we call the Create method
  ssTmp := nil;
  ssTmp2 := nil;
  ssTmp := TInheritableSingleton.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  ssTmp2 := TInheritableSingleton.Create;
  Check(ssTmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
end;

procedure TInheritableSingletonTest.TestFreeError;
var
  ssTmp, ssTmp2: TInheritableSingleton;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  ssTmp := nil;
  // Get references to the singleton
  ssTmp := TInheritableSingleton.Create;
  ssTmp2 := TInheritableSingleton.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
  Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
  Check(ssTmp = ssTmp2, 'Singleton references are not the same');
  // Now call the free method. Should get an EAbort exception
  try
    ssTmp.Free;
    fail('EAbort exception not raised');
  except
    on E: Exception do
    begin
      if E.ClassType <> EAbort then
        raise;
    end
  end;
  ssTmp := nil;
  // Try to reference the singleton's property
  ssTmp2.Value := 100;
  // Get another reference to the singleton
  ssTmp := TInheritableSingleton.Create;
  // Check that we've still got the value
  Check(ssTmp.Value = 100, 'Value not retained');
end;

procedure TInheritableSingletonTest.TestNotNil;
var
  ssTmp: TInheritableSingleton;
begin
  // We should be able to get the singleton from the function
  ssTmp := nil;
  ssTmp := TInheritableSingleton.Create;
  Check(Assigned(ssTmp), 'ssTmp not assigned');
end;

procedure TInheritableSingletonTest.TestCreateDescendant;
var
  isTmp: TInheritableSingleton;
  isd1Tmp, isd1Tmp2: TInheritableSingletonDescendant1;
begin
  // See what happens when we create a descendant of TInheritableSingleton
{  // First, make sure the singleton doesn't exist
  isTmp := TInheritableSingleton.Create;         // Get a reference to it
  if Assigned(isTmp) then
    isTmp.DestroyInstance;                     // Destroy it}
  // Now do the test. Create two instances of the descendant type, and check
  // that both variables point to the same instance
  isd1Tmp := nil;
  isd1Tmp2 := nil;
  isd1Tmp := TInheritableSingletonDescendant1.Create;
  Check(isd1Tmp.InitCount = 1, 'InitCount is not 1');
  isd1Tmp2 := TInheritableSingletonDescendant1.Create;
  Check(isd1Tmp.InitCount = 1, 'InitCount is not 1');
  Check(Assigned(isd1Tmp), 'isd1Tmp not assigned');
  Check(Assigned(isd1Tmp2), 'isd1Tmp2 not assigned');
  Check(isd1Tmp = isd1Tmp2, 'Singleton references are not the same');
end;

procedure TInheritableSingletonTest.TestCreateDescendants;
var
  isTmp: TInheritableSingleton;
  isd1Tmp: TInheritableSingletonDescendant1;
  isd2Tmp: TInheritableSingletonDescendant2;
begin
  // See what happens when we create descendants of more than one type
  // First, make sure the singleton doesn't exist
{  isTmp := TInheritableSingleton.Create;         // Get a reference to it
  if Assigned(isTmp) then
    isTmp.DestroyInstance;                     // Destroy it}
  // Now do the test
  // Create a descendant of the first type and check its class name
  isd1Tmp := nil;
  isd1Tmp := TInheritableSingletonDescendant1.Create;
  Check(isd1Tmp.ClassName = 'TInheritableSingletonDescendant1',
                             'isd1Tmp is the wrong class (' +
                             isd1Tmp.ClassName + ')');
  // Create a descendant of the second type and check its class name
  isd2Tmp := nil;
  isd2Tmp := TInheritableSingletonDescendant2.Create;
  Check(isd2Tmp.ClassName = 'TInheritableSingletonDescendant2',
                             'isd2Tmp is the wrong class (' +
                             isd2Tmp.ClassName + ')');
end;

{ TInheritableSingletonDescendant3 }

constructor TInheritableSingletonDescendant3.Create;
begin
  FMyValue2 := 0;
  inherited;
end;

procedure TInheritableSingletonTest.TestOverriddenCreate;
var
  siod3Tmp, siod3Tmp2: TInheritableSingletonDescendant3;
begin
  // Check that overridden Create gets called every time
  try
    siod3Tmp := TInheritableSingletonDescendant3.Create;
    siod3Tmp.MyValue2 := 100;
    siod3Tmp2 := TInheritableSingletonDescendant3.Create;
    Check(siod3Tmp2.MyValue2 = 0, 'Overridden Create code was not called twice');
  finally
    if Assigned(siod3Tmp) then
      siod3Tmp.DestroyInstance;
  end;
end;

procedure TInheritableSingletonDescendant3.Initialise;
begin
  FMyValue2 := 10;
end;

procedure TInheritableSingletonTest.TestOverriddenInitialise;
var
  siod3Tmp, siod3Tmp2: TInheritableSingletonDescendant3;
begin
  // Check that overridden Initialise only gets called once
  try
    siod3Tmp := TInheritableSingletonDescendant3.Create;
    Check(siod3Tmp.MyValue2 = 10, 'Overridden Initialise code was not called');
    siod3Tmp.MyValue2 := 100;
    siod3Tmp2 := TInheritableSingletonDescendant3.Create;
    Check(siod3Tmp2.MyValue2 = 0, 'Overridden Initialise code was called twice');
  finally
    if Assigned(siod3Tmp) then
      siod3Tmp.DestroyInstance;
  end;
end;

initialization
begin
  TestFramework.RegisterTest(TSimpleSingletonTest.Suite);
  TestFramework.RegisterTest(TSimpleSingleton2Test.Suite);
  TestFramework.RegisterTest(TSimpleSingleton3Test.Suite);
  TestFramework.RegisterTest(TSimpleSingleton4Test.Suite);
  TestFramework.RegisterTest(TInheritableSingletonTest.Suite);
end;

end.
