unit uSingletonInheritOnceTest;

interface

uses
  TestFramework
  , uTestCaseException
  , SingletonInheritOnce
  ;

type

  TSingletonInheritOnceTest = class(TTestCaseException)
  private

  published
    procedure TestNotNil;
    procedure TestCreate;
    procedure TestFree;
    procedure TestCreateDescendant;
    procedure TestCreateDescendants;
  end;

implementation

uses
  sysutils                        // for definition of EAbort
  ;

{ TSingletonInheritOnceTest }

{ Create two descendant classes }

type
  TSingletonInheritOnceDescendant1 = class(TSingleton)
  private
    FMyValue1: integer;
  public
    property MyValue1: integer
      read FMyValue1
      write FMyValue1;
  end;

  TSingletonInheritOnceDescendant2 = class(TSingleton)
  private
    FMyValue2: integer;
  public
    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

procedure TSingletonInheritOnceTest.TestCreate;
var
  ssTmp, ssTmp2: TSingleton;
begin
  // See what happens when we call the Create method
  try
    ssTmp := nil;
    ssTmp2 := nil;
    ssTmp := TSingleton.Create;
    ssTmp2 := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
    Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
    Check(ssTmp = ssTmp2, 'Singleton references are not the same');
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritOnceTest.TestFree;
var
  ssTmp, ssTmp2: TSingleton;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  try
    Self.SetExpectedException(EAbort);
    ssTmp := nil;
    // Get references to the singleton
    ssTmp := TSingleton.Create;
    ssTmp2 := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
    Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
    Check(ssTmp = ssTmp2, 'Singleton references are not the same');
    // Now call the free method. Should get an EAbort exception. Test will fail if
    // EAbort not raised
    ssTmp.Free;
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritOnceTest.TestNotNil;
var
  ssTmp: TSingleton;
begin
  // We should be able to get the singleton from the function
  try
    ssTmp := nil;
    ssTmp := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritOnceTest.TestCreateDescendant;
var
  sioTmp: TSingleton;
  siod1Tmp, siod1Tmp2: TSingletonInheritOnceDescendant1;
begin
  // See what happens when we create a descendant of TSingletonInheritOnce
  // Create two instances of the descendant type, and check
  // that both variables point to the same instance
  try
    siod1Tmp := nil;
    siod1Tmp2 := nil;
    siod1Tmp := TSingletonInheritOnceDescendant1.Create;
    Check(siod1Tmp.ClassType = TSingletonInheritOnceDescendant1,
                 'siod1Tmp is the wrong class type (' + siod1Tmp.ClassName + ')');
    siod1Tmp.MyValue1 := 100;
    siod1Tmp2 := TSingletonInheritOnceDescendant1.Create;
    Check(Assigned(siod1Tmp), 'siod1Tmp not assigned');
    Check(Assigned(siod1Tmp2), 'siod1Tmp2 not assigned');
    Check(siod1Tmp = siod1Tmp2, 'Singleton references are not the same');
    Check(siod1Tmp.MyValue1 = siod1Tmp2.MyValue1,
                                          'Singleton values are not the same');
  finally
    if Assigned(siod1Tmp) then
      siod1Tmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritOnceTest.TestCreateDescendants;
var
  sioTmp: TSingleton;
  siod1Tmp: TSingletonInheritOnceDescendant1;
  siod2Tmp: TSingletonInheritOnceDescendant2;
begin
  // See what happens when we create descendants of more than one type
  // We should get an ESingletonInheritOnce exception - if we get one, the test will
  // pass.
  try
    Self.SetExpectedException(ESingleton);
    // Create a descendant of the first type and check its class name
    siod1Tmp := nil;
    siod1Tmp := TSingletonInheritOnceDescendant1.Create;
    Check(siod1Tmp.ClassName = 'TSingletonInheritOnceDescendant1',
                               'siod1Tmp is the wrong class (' +
                               siod1Tmp.ClassName + ')');
    // Create a descendant of the second type and check its class name
    siod2Tmp := nil;
    // The next line should throw an ESingletonInheritOnce exception
    siod2Tmp := TSingletonInheritOnceDescendant2.Create;
    Check(siod2Tmp.ClassName = 'TSingletonInheritOnceDescendant2',
                               'siod2Tmp is the wrong class (' +
                               siod2Tmp.ClassName + ')');
  finally
    if Assigned(siod1Tmp) then
      siod1Tmp.DestroyInstance;
  end;
end;

initialization
  TestFramework.RegisterTest(TSingletonInheritOnceTest.Suite);

end.
