unit uSingletonInheritManyTest;

interface

uses
  TestFramework
  , uTestCaseException
  , SingletonInheritMany
  ;

type

  TSingletonInheritManyTest = class(TTestCaseException)
  private

  published
    procedure TestNotNil;
    procedure TestCreate;
    procedure TestFree;
    procedure TestCreateDescendant;
    procedure TestCreateDescendants;
    procedure TestOverriddenCreate;
    procedure TestOverriddenInitialise;
  end;

implementation

uses
  sysutils                        // for definition of EAbort
  ;

{ TSingletonInheritManyTest }

{ Create two descendant classes }

type
  TSingletonInheritManyDescendant1 = class(TSingleton)
  private
    FMyValue1: integer;
  public
    property MyValue1: integer
      read FMyValue1
      write FMyValue1;
  end;

  TSingletonInheritManyDescendant2 = class(TSingleton)
  private
    FMyValue2: integer;
  public
    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

  TSingletonInheritManyDescendant3 = class(TSingleton)
  private
    FMyValue2: integer;
  protected
    procedure Initialise; override;
  public

    constructor Create; override;

    property MyValue2: integer
      read FMyValue2
      write FMyValue2;
  end;

procedure TSingletonInheritManyTest.TestCreate;
var
  ssTmp, ssTmp2: TSingleton;
begin
  // See what happens when we call the Create method
  try
    ssTmp := nil;
    ssTmp2 := nil;
    ssTmp := TSingleton.Create;
    ssTmp2 := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
    Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
    Check(ssTmp = ssTmp2, 'Singleton references are not the same');
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritManyTest.TestFree;
var
  ssTmp, ssTmp2: TSingleton;
  nTmp: integer;
begin
  // See what happens when we call the singleton's free method
  try
    Self.SetExpectedException(EAbort);
    ssTmp := nil;
    // Get references to the singleton
    ssTmp := TSingleton.Create;
    ssTmp2 := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
    Check(Assigned(ssTmp2), 'ssTmp2 not assigned');
    Check(ssTmp = ssTmp2, 'Singleton references are not the same');
    // Now call the free method. Should get an EAbort exception. Test will fail if
    // EAbort not raised
    ssTmp.Free;
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritManyTest.TestNotNil;
var
  ssTmp: TSingleton;
begin
  // We should be able to get the singleton from the function
  try
    ssTmp := nil;
    ssTmp := TSingleton.Create;
    Check(Assigned(ssTmp), 'ssTmp not assigned');
  finally
    if Assigned(ssTmp) then
      ssTmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritManyTest.TestCreateDescendant;
var
  sioTmp: TSingleton;
  siod1Tmp, siod1Tmp2: TSingletonInheritManyDescendant1;
begin
  // See what happens when we create a descendant of TSingletonInheritMany
  // Create two instances of the descendant type, and check
  // that both variables point to the same instance
  try
    siod1Tmp := nil;
    siod1Tmp2 := nil;
    siod1Tmp := TSingletonInheritManyDescendant1.Create;
    Check(siod1Tmp.ClassType = TSingletonInheritManyDescendant1,
                 'siod1Tmp is the wrong class type (' + siod1Tmp.ClassName + ')');
    siod1Tmp.MyValue1 := 100;
    siod1Tmp2 := TSingletonInheritManyDescendant1.Create;
    Check(Assigned(siod1Tmp), 'siod1Tmp not assigned');
    Check(Assigned(siod1Tmp2), 'siod1Tmp2 not assigned');
    Check(siod1Tmp = siod1Tmp2, 'Singleton references are not the same');
    Check(siod1Tmp.MyValue1 = siod1Tmp2.MyValue1,
                                          'Singleton values are not the same');
  finally
    if Assigned(siod1Tmp) then
      siod1Tmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritManyTest.TestCreateDescendants;
var
  sioTmp: TSingleton;
  siod1Tmp: TSingletonInheritManyDescendant1;
  siod2Tmp: TSingletonInheritManyDescendant2;
begin
  // See what happens when we create descendants of more than one type
  // It should create singletons of different types successfully
  try
    // Create a descendant of the first type and check its class name
    siod1Tmp := nil;
    siod1Tmp := TSingletonInheritManyDescendant1.Create;
    Check(siod1Tmp.ClassName = 'TSingletonInheritManyDescendant1',
                               'siod1Tmp is the wrong class (' +
                               siod1Tmp.ClassName + ')');
    // Create a descendant of the second type and check its class name
    siod2Tmp := nil;
    // The next line should throw an ESingletonInheritMany exception
    siod2Tmp := TSingletonInheritManyDescendant2.Create;
    Check(siod2Tmp.ClassName = 'TSingletonInheritManyDescendant2',
                               'siod2Tmp is the wrong class (' +
                               siod2Tmp.ClassName + ')');
  finally
    if Assigned(siod1Tmp) then
      siod1Tmp.DestroyInstance;
  end;
end;

{ TSingletonInheritManyDescendant3 }

constructor TSingletonInheritManyDescendant3.Create;
begin
  FMyValue2 := 0;
  inherited;
end;

procedure TSingletonInheritManyDescendant3.Initialise;
begin
  FMyValue2 := 10;
end;

procedure TSingletonInheritManyTest.TestOverriddenCreate;
var
  simd3Tmp, simd3Tmp2: TSingletonInheritManyDescendant3;
begin
  // Check that overridden Create gets called every time
  try
    simd3Tmp := TSingletonInheritManyDescendant3.Create;
    simd3Tmp.MyValue2 := 100;
    simd3Tmp2 := TSingletonInheritManyDescendant3.Create;
    Check(simd3Tmp2.MyValue2 = 0, 'Overridden Create code was not called twice');
  finally
    if Assigned(simd3Tmp) then
      simd3Tmp.DestroyInstance;
  end;
end;

procedure TSingletonInheritManyTest.TestOverriddenInitialise;
var
  simd3Tmp, simd3Tmp2: TSingletonInheritManyDescendant3;
begin
  // Check that overridden Initialise only gets called once
  try
    simd3Tmp := TSingletonInheritManyDescendant3.Create;
    Check(simd3Tmp.MyValue2 = 10, 'Overridden Initialise code was not called');
    simd3Tmp.MyValue2 := 100;
    simd3Tmp2 := TSingletonInheritManyDescendant3.Create;
    Check(simd3Tmp2.MyValue2 = 0, 'Overridden Initialise code was called twice');
  finally
    if Assigned(simd3Tmp) then
      simd3Tmp.DestroyInstance;
  end;
end;

initialization
  TestFramework.RegisterTest(TSingletonInheritManyTest.Suite);

end.
