{***************************************************************
 *
 * Unit Name: uSingleton
 * Purpose  : To define a base Singleton class
 * Author   : Rob Bracken
 * History  : 11/3/2002 - class created
 *
 ****************************************************************}

unit uSingleton;

interface

type

  TSingleton = class
  private

  protected

    // These methods are abstract to remind the class designer about them.
    // Put in these any code that should be executed just once (such as creation
    // of TList objects
    procedure OnCreate(Sender: TObject); virtual; abstract;
    procedure OnDestroy(Sender: TObject); virtual; abstract;

  public

    // Put in these any code that should be executed on every call (such as
    // registering objects with the Singleton)
    constructor Create; virtual;
    destructor Destroy; override;

    procedure FreeInstance; override;
    class function NewInstance: TObject; override;

  end;

implementation

uses
  sysutils          // for definition of EAbstractError
  ;

{ TSingleton }

// As Delphi doesn't support true class variables, we have to use unit variables
var
  UFSingleton: TSingleton = nil;    // initialise it to Unassigned
  UFRefCount: integer = 0;          // initialise reference count to zero

const
  EAbstractErrorMessage =
                        'You must override the OnCreate and OnDestroy methods!';

//******************* TSingleton.Create *************************

constructor TSingleton.Create;
begin
  // Fire the OnCreate event, but only if we've just been created
  if (UFRefCount = 1) then
  begin
    try
      OnCreate(Self);
    except
      // If the programmer hasn't overridden OnCreate, display a meaningful
      // error message
      on E: EAbstractError do
        // Change the error message
        raise EAbstractError.Create('EAbstractError from ' + Self.ClassName +
              #13#10 + EAbstractErrorMessage);
    end;
  end;
end;

//******************* TSingleton.Destroy *************************

destructor TSingleton.Destroy;
begin
  // Fire the OnDestroy event, but only if we're about to be destroyed
  if (UFRefCount = 1) then
  begin
    try
      OnDestroy(Self);
    except
      // If the programmer hasn't overridden OnDestroy, display a meaningful
      // error message
      on E: EAbstractError do
        // Change the error message
        raise EAbstractError.Create('EAbstractError from ' + Self.ClassName +
              #13#10 + EAbstractErrorMessage);
    end;
  end;
  inherited;
end;

//******************* TSingleton.FreeInstance *************************

procedure TSingleton.FreeInstance;
begin
  // Decrement the reference counter
  dec(UFRefCount);
  // If the reference counter reaches zero, free the object
  if (UFRefCount = 0) then
  begin
    // Cleanup and deallocate memory
    inherited;        // You can verify that the object is freed by using Debug DCUs
    UFSingleton := nil;       // Clear this, in case we create some more
  end;
end;

//******************* TSingleton.NewInstance *************************

class function TSingleton.NewInstance: TObject;
begin
  // If we've already created the singleton, return it, otherwise create a new one
  if Assigned(UFSingleton) then
  begin
    // Singleton already created - return it
    Result := UFSingleton;
  end
  else
  begin
    // Singleton not created yet - create and return it
    // Get memory for the new object instance
    GetMem(Pointer(UFSingleton), Self.InstanceSize);
    // Initialise the object's memory block to all zeroes
    Result := InitInstance(UFSingleton);
  end;
  // Maintain a reference counter, so we know when we've finished with the singleton
  inc(UFRefCount);
end;

end.
