{***************************************************************
 *
 * Unit Name: SingletonInheritOnce
 * Purpose  : Base class for a singleton object (1 descendant only)
 * Author   : Rob Bracken for uk-bug developer's magazine 3rd May 2002
 * History  :
 *
 * You are free to use this in your own projects
 *
 * Notes:
 *
 * 1. Code in overridden Create methods is executed every time Create is called
 * 2. Code in overridden Initialise methods is only ever executed once
 * 3. If you try to get references to more than one descendant type, you'll get
 *    a reference to the first type, but when you try to get a reference to the
 *    second and subsequent types, you'll get an ESingleton error. Use the
 *    base class in the SingletonInheritMany unit if you need more than one
 *    Singleton descendant type.
 *
 ****************************************************************}

unit SingletonInheritOnce;

interface

uses
  sysutils            // for definition of Exception
  ;

type
  ESingleton = class(Exception);

  TSingleton = class
  private

    FInitialised: boolean;

{$IFNDEF TESTING}
    destructor DestroyInstance;
{$ENDIF}

  protected

    procedure Initialise; virtual;

  public

    constructor Create;
    destructor Destroy; override;
{$IFDEF TESTING}
    destructor DestroyInstance;
{$ENDIF}

    procedure FreeInstance; override;
    class function NewInstance: TObject; override;

  end;

implementation

uses
  Dialogs             // for ShowMessage
  ;

var
  gSingleton: TSingleton = nil;

{ TSingleton }

constructor TSingleton.Create;
begin
  if not FInitialised then
  begin
    Initialise;
  end;
  FInitialised := True;
end;

destructor TSingleton.Destroy;
begin
  Abort;
end;

destructor TSingleton.DestroyInstance;
begin
  inherited Destroy;
end;

procedure TSingleton.FreeInstance;
begin
  if Assigned(gSingleton) then
  begin
    inherited;
    // Set the unit variable to nil, in case we need it again
    gSingleton := nil;
  end;
end;

procedure TSingleton.Initialise;
begin
  // do initialisation here
end;

class function TSingleton.NewInstance: TObject;
begin
  // If we've already created the singleton, return it, otherwise create a new one
  if Assigned(gSingleton) then
  begin
    // Singleton already created - test its class and return it
    if Self = gSingleton.ClassType then
    begin
      Result := gSingleton;
    end
    else
    begin
      raise ESingleton.Create(
                          'Singleton is already created with a different type');
    end;
  end
  else
  begin
    // Singleton not created yet - create and return it
    // Get memory for the new object instance
    GetMem(Pointer(gSingleton), Self.InstanceSize);
    // Initialise the object's memory block to all zeroes
    Result := InitInstance(gSingleton);
  end;
end;

initialization
// Dummy initialization, so we can have a finalization

finalization
  if Assigned(gSingleton) then
    gSingleton.DestroyInstance;

end.
