{***************************************************************
 *
 * Unit Name: SingletonInheritMany
 * Purpose  : Base class for Singletons (multiple descendants possible)
 * Author   : Rob Bracken for uk-bug developer's magazine 3rd May 2002
 * History  :
 *
 * You are free to use this in your own projects
 *
 * Notes:
 *
 * 1. Code in overridden Create methods is executed every time Create is called
 * 2. Code in overridden Initialise methods is only ever executed once
 *
 ****************************************************************}

unit SingletonInheritMany;

interface

uses
  sysutils            // for definition of Exception
  ;

type
  ESingleton = class(Exception);

  TSingleton = class
  private

    FInitialised: boolean;
    FInitCount: integer;          // for testing purposes
    FValue: integer;              // for testing purposes

{$IFNDEF TESTING}
    destructor DestroyInstance;
{$ENDIF}

  protected

    procedure Initialise; virtual;

  public

    constructor Create; virtual;
    destructor Destroy; override;

{$IFDEF TESTING}
    destructor DestroyInstance;
{$ENDIF}

    procedure FreeInstance; override;
    class function NewInstance: TObject; override;

    property Value: integer         // for testing purposes
      read FValue
      write FValue;

    property InitCount: integer     // for testing purposes
      read FInitCount;

  end;

implementation

uses
  Dialogs             // for ShowMessage
  , classes           // for TList
  ;

var
  gSingletonList: TList = nil;

{ TSingleton }

constructor TSingleton.Create;
begin
  if not FInitialised then
  begin
    Initialise;
  end;
  FInitialised := True;
end;

destructor TSingleton.Destroy;
begin
  Abort;
end;

destructor TSingleton.DestroyInstance;
begin
  inherited Destroy;
end;

procedure TSingleton.FreeInstance;
var
  nSingleton: integer;
begin
{// Uncomment this to see when FreeInstance is called
  ShowMessage('TSingleton.FreeInstance');}
  // Find the relevant instance in the TList
  if Assigned(gSingletonList) then
  begin
    for nSingleton := 0 to Pred(gSingletonList.Count) do
    begin
      if TObject(gSingletonList.Items[nSingleton]).ClassType = Self.ClassType then
      begin
        inherited;
        // Remove the pointer from the TList
        gSingletonList.Delete(nSingleton);
        Exit;               // Avoid "List index out of bounds"
      end;
    end;
  end;
end;

procedure TSingleton.Initialise;
begin
  // do initialisation here

  // Keep track of how many times we've called the Create code, so we can test it
  Inc(FInitCount);
end;

class function TSingleton.NewInstance: TObject;
var
  nSingleton: integer;
  ptrObject: pointer;
begin
{// Uncomment this to see when NewInstance is called
  ShowMessage('TSingleton.NewInstance');}
  // If we've already created the singleton, return it, otherwise create a new one
  // Look for an existing instance in the TList. Note that Self here refers to a
  // class, rather than an object
  for nSingleton := 0 to Pred(gSingletonList.Count) do
  begin
    if Self = TObject(gSingletonList.Items[nSingleton]).ClassType then
    begin
      // Found an existing instance, return it and exit
      Result := TObject(gSingletonList.Items[nSingleton]);
      Exit;
    end;
  end;
  // Singleton not created yet - create and return it
  // Get memory for the new object instance
  GetMem(ptrObject, Self.InstanceSize);
  // Initialise the object's memory block to all zeroes
  Result := InitInstance(ptrObject);
  // Save the pointer in the TList
  gSingletonList.Add(ptrObject);
end;

initialization
  gSingletonList := TList.Create;

finalization
  if Assigned(gSingletonList) then
  begin
    while gSingletonList.Count > 0 do
    begin
      TSingleton(gSingletonList.Items[0]).DestroyInstance;
      // DestroyInstance will delete the current item from gSingletonList
    end;
    gSingletonList.Free;
  end;

end.
