using System;

using Android.App;
using Android.Content;
using Android.OS;
using Android.Views;
using Android.Widget;

using Android.Database;
using Android.Util;

namespace TimeSheet
{
    [Activity(Label = "@string/NewEntryTitle", WindowSoftInputMode = SoftInput.StateHidden)]
    public class NewEntryActivity : Activity
    {
        private Spinner clientSpinner;
        private EditText rateEditText;
        private EditText startTimeEditText;
        private EditText endTimeEditText;
        private EditText commentEditText;
        private ICursor cursor;
        private Client currentClient;
        private Entry newEntry;

        private const int TIME_DIALOG_START = 1;
        private const int TIME_DIALOG_END = 2;
        private const int SUBMIT_CONFIRM_DIALOG = 3;

        protected override void OnCreate(Bundle bundle)
        {
            base.OnCreate(bundle);
            try
            {
                SetContentView(Resource.Layout.NewEntryLayout);
                clientSpinner = FindViewById<Spinner>(Resource.Id.NewEntryClientSpinner);
                cursor = ((TimeSheetApplication)Application).DatabaseHelper.FetchAllClientsCursor();
                StartManagingCursor(cursor);
                var adapter = new SimpleCursorAdapter(this,
                    Android.Resource.Layout.SimpleSpinnerItem, cursor,
                    new string[] { "name" }, new int[] { Android.Resource.Id.Text1 });
                adapter.SetDropDownViewResource(Android.Resource.Layout.SimpleSpinnerDropDownItem);
                clientSpinner.Adapter = adapter;
                clientSpinner.ItemSelected += new EventHandler<ItemEventArgs>(clientSpinner_ItemSelected);
                rateEditText = FindViewById<EditText>(Resource.Id.NewEntryRateEditText);
                startTimeEditText = FindViewById<EditText>(Resource.Id.NewEntryStartTimeEditText);
                startTimeEditText.Touch = TimeEditTextTouchHandler;
                endTimeEditText = FindViewById<EditText>(Resource.Id.NewEntryEndTimeEditText);
                endTimeEditText.Touch = TimeEditTextTouchHandler;
                commentEditText = FindViewById<EditText>(Resource.Id.NewEntryCommentEditText);
                ClearUI();
                Button submitButton = FindViewById<Button>(Resource.Id.NewEntrySubmitButton);
                submitButton.Click += new EventHandler(submitButton_Click);
                Button startNowButton = FindViewById<Button>(Resource.Id.NewEntryStartNowButton);
                startNowButton.Click += delegate
                {
                    startTimeEditText.Text = DateTime.Now.ToString(Consts.TimeFmt);
                };
                Button endNowButton = FindViewById<Button>(Resource.Id.NewEntryEndNowButton);
                endNowButton.Click += delegate
                {
                    endTimeEditText.Text = DateTime.Now.ToString(Consts.TimeFmt);
                };
            }
            catch (Java.Lang.Throwable t)
            {
                Common.LogException(t);
                throw;
            }
        }

        protected override void OnPause()
        {
            Log.Info(Consts.Tag, "NewEntryActivity pausing");
            base.OnPause();
            Log.Info(Consts.Tag, "NewEntryActivity paused");            
        }

        protected override void OnResume()
        {
            Log.Info(Consts.Tag, "NewEntryActivity resuming");
            base.OnResume();
            Log.Info(Consts.Tag, "NewEntryActivity resumed");            
        }

        protected override void OnStart()
        {
            Log.Info(Consts.Tag, "NewEntryActivity starting");
            base.OnStart();
            Log.Info(Consts.Tag, "NewEntryActivity started");
        }

        protected override void OnStop()
        {
            Log.Info(Consts.Tag, "NewEntryActivity stopping");
            base.OnStop();
            Log.Info(Consts.Tag, "NewEntryActivity stopped");
        }

        private void ClearUI()
        {
            startTimeEditText.Text = DateTime.Now.ToString(Consts.TimeFmt);
            endTimeEditText.Text = DateTime.Now.ToString(Consts.TimeFmt);
            commentEditText.Text = "";
            if (currentClient != null)
                rateEditText.Text = currentClient.DefaultHourlyRate.ToString();
        }

        void clientSpinner_ItemSelected(object sender, ItemEventArgs e)
        {
            currentClient = ((TimeSheetApplication)Application).DatabaseHelper.FetchClient((int)e.Id);
            Common.ShortToast(GetString(Resource.String.NewEntryClientSelectedToastFmt),
                currentClient.Name, currentClient.DefaultHourlyRate, currentClient.Comment);
            //TODO: bug remains that on resumption of the activity, the ItemSelected handler kicks in and 
            //overwrites the rate, which _may_ have been customised
            Log.Info(Consts.Tag, "About to overwrite the rate value");
            rateEditText.Text = currentClient.DefaultHourlyRate.ToString();
        }

        void submitButton_Click(object sender, EventArgs e)
        {
            newEntry = new Entry();
            TimeSheetPreferences prefs = new TimeSheetPreferences(ApplicationContext);
            newEntry.UserID = prefs.UserID;
            newEntry.ClientID = currentClient.ID;
            double rate;
            if (!double.TryParse(rateEditText.Text, out rate))
            {
                Common.LongToast(Resource.String.NewEntryBadRateValue);
                return;
            }
            newEntry.HourlyRate = rate;
            newEntry.StartTime = Convert.ToDateTime(startTimeEditText.Text);
            newEntry.EndTime = Convert.ToDateTime(endTimeEditText.Text);
            if ((startTimeEditText.Text == endTimeEditText.Text) || (newEntry.EndTime <= newEntry.StartTime))
            {
                Common.LongToast(Resource.String.NewEntryBadTimeValue);
                return;
            }
            newEntry.Comment = commentEditText.Text;
            newEntry.Synced = false;

            //Final confirmation before submission of timesheet entry
            string confirm = string.Format(GetString(Resource.String.NewEntrySubmitConfirmationFmt),
                prefs.UserName, prefs.UserEmail, currentClient.Name,
                newEntry.StartTime, newEntry.EndTime, newEntry.HourlyRate, newEntry.Comment);
            //Put the target confirmation message in a bundle and pass through to the dialog handler
            Bundle b = new Bundle();
            b.PutString("msg", confirm);
            ShowDialog(SUBMIT_CONFIRM_DIALOG, b);
        }

        private bool TimeEditTextTouchHandler(View v, MotionEvent e)
        {
            if ((v is EditText) && (e.Action == MotionEventActions.Up))
            {
                EditText edt = (EditText)v;
                //Put current time from EditText in a bundle to pass along to the dialog handler
                ShowDialog(v == startTimeEditText ? TIME_DIALOG_START : TIME_DIALOG_END);
                return true;
            }
            return false;
        }
        
        protected override Dialog OnCreateDialog(int id, Bundle args)
        {
            //Do the initial creation of the dialogs as required
            TimePickerDialog timeDialog = null;
            switch (id)
            {
                case TIME_DIALOG_START:
                    timeDialog = new TimePickerDialog(
                        this, (o, e) =>
                        {
                            startTimeEditText.Text = string.Format("{0:00}:{1:00}", e.HourOfDay, e.Minute);
                        },
                        Convert.ToInt32(startTimeEditText.Text.Substring(0, 2)),
                        Convert.ToInt32(startTimeEditText.Text.Substring(3, 2)), true);
                    return timeDialog;
                case TIME_DIALOG_END:
                    timeDialog = new TimePickerDialog(
                        this, (o, e) =>
                        {
                            endTimeEditText.Text = string.Format("{0:00}:{1:00}", e.HourOfDay, e.Minute);
                        },
                        Convert.ToInt32(endTimeEditText.Text.Substring(0, 2)),
                        Convert.ToInt32(endTimeEditText.Text.Substring(3, 2)), true);
                    return timeDialog;
                case SUBMIT_CONFIRM_DIALOG:
                    var builder = new AlertDialog.Builder(this);
                    builder.SetMessage(args.GetString("msg"))
                           .SetTitle(Resource.String.NewEntrySubmitConfirmationTitle)
                           .SetIcon(Android.Resource.Drawable.IcDialogAlert)
                           .SetCancelable(true)
                           .SetPositiveButton(Android.Resource.String.Ok,
                             (s, a) => 
                             {
                                 ((TimeSheetApplication)Application).DatabaseHelper.AddTimeSheetEntry(newEntry);
                                 Common.ShortToast(Resource.String.NewEntrySubmittedToast);
                                 newEntry = null;
                                 ClearUI(); //reset controls
                             })
                           .SetNegativeButton(Android.Resource.String.Cancel,
                             (IDialogInterfaceOnClickListener)null);
                    return builder.Create();
            }
            return null;
        }

        protected override void OnPrepareDialog(int id, Dialog dialog)
        {
            //Update the time displayed in the time picker dialog each subsequent time it is displayed
            TimePickerDialog timeDialog;
            switch (id)
            {
                case TIME_DIALOG_START:
                    timeDialog = (TimePickerDialog)dialog;
                    timeDialog.UpdateTime(
                        Convert.ToInt32(startTimeEditText.Text.Substring(0, 2)),
                        Convert.ToInt32(startTimeEditText.Text.Substring(3, 2)));
                    break;
                case TIME_DIALOG_END:
                    timeDialog = (TimePickerDialog)dialog;
                    timeDialog.UpdateTime(
                        Convert.ToInt32(endTimeEditText.Text.Substring(0, 2)),
                        Convert.ToInt32(endTimeEditText.Text.Substring(3, 2)));
                    break;
            }
        }
    }
}