﻿using System;

using Android.App;
using Android.Content;
using Android.Views;
using Android.Widget;
using Android.OS;

using Android.Util;
using System.Threading;
using System.Text;

//Notes
/*
 * Stress test the app using Monkey from the adb shell, e.g.:
 * 
 * adb -e shell monkey -p com.blong.TimeSheet -v -v 500
 */

namespace TimeSheet
{
    [Activity(Label = "@string/ApplicationName")]
    public class MainActivity : Activity
    {
        //Menu items
        private const int ABOUT_ID = 0;
        private const int EXIT_ID = 1;
        //Launched activity result codes
        private const int ACTIVITY_RESULT_LOGIN = 1;
        //Dialog codes
        private const int SYNC_CONFIRM_DIALOG = 1;

        private TimeSheetPreferences prefs;
        private TextView loginDetailsLabel;
        private ProgressDialog progressDialog;

        protected override void OnCreate(Bundle bundle)
        {
            base.OnCreate(bundle);
            prefs = new TimeSheetPreferences(ApplicationContext);

            // Set our view from the "main" layout resource
            SetContentView(Resource.Layout.MainLayout);

            Button newEntryButton = FindViewById<Button>(Resource.Id.MainNewEntryButton);
            newEntryButton.Click += delegate
            {
                StartActivity(new Intent(this, typeof(NewEntryActivity)));
            };
            Button syncButton = FindViewById<Button>(Resource.Id.MainSyncButton);
            syncButton.Click += delegate
            {
                ShowDialog(SYNC_CONFIRM_DIALOG);
            };
            Button reportButton = FindViewById<Button>(Resource.Id.MainReportButton);
            reportButton.Click += delegate
            {
                StartActivity(new Intent(this, typeof(ReportActivity)));
            };
            Button switchUserButton = FindViewById<Button>(Resource.Id.MainSwitchUserButton);
            switchUserButton.Click += delegate
            {
                StartActivityForResult(new Intent(this, typeof(LoginActivity)), ACTIVITY_RESULT_LOGIN);
            };
            loginDetailsLabel = FindViewById<TextView>(Resource.Id.MainLoginDetailsLabel);
        }

        protected override void OnResume()
        {
            base.OnResume();
            //After the login activity has been successfully passed, we should come back through here and be ok
            if (prefs.UserName == Consts.NA)
                StartActivityForResult(new Intent(this, typeof(LoginActivity)), ACTIVITY_RESULT_LOGIN);
            else
                loginDetailsLabel.Text = string.Format("{0} ({1})", prefs.UserName, prefs.UserEmail);
        }

        public override bool OnCreateOptionsMenu(IMenu menu)
        {
            //NB: options menu items support icons, unlike context menu
            var item = menu.Add(0, ABOUT_ID, 0, Resource.String.MainAboutMenu);
            item.SetIcon(Android.Resource.Drawable.IcMenuInfoDetails);
            //Guidelines actually suggest an exit option is not necessary/appropriate, but...
            item = menu.Add(0, EXIT_ID, 0, Resource.String.MainExitMenu);
            item.SetIcon(Android.Resource.Drawable.IcMenuCloseClearCancel);
            return true;
        }

        public override bool OnOptionsItemSelected(IMenuItem item)
        {
            switch (item.ItemId)
            {
                case ABOUT_ID:
                    StartActivity(new Intent(this, typeof(AboutActivity)));
                    break;
                case EXIT_ID:
                    //No need for validation as the activity can be switched back to, and the user data is in the database
                    Finish();
                    break;

            }
            return false;
        }

        protected override void OnActivityResult(int requestCode, Result resultCode, Intent data)
        {
            base.OnActivityResult(requestCode, resultCode, data);
            switch (requestCode)
            {
                case ACTIVITY_RESULT_LOGIN:
                    //If login tells us we are no longer required (Back was pressed and we have no successful login), then bail out
                    if ((resultCode == Result.Canceled) && ((prefs.UserName == Consts.NA)))
                        Finish();
                    break;
            }
        }

        protected override Dialog OnCreateDialog(int id)
        {
            if (id == SYNC_CONFIRM_DIALOG)
            {
                var builder = new AlertDialog.Builder(this);
                builder.SetMessage(Resource.String.MainSyncConfirmationMessage)
                       .SetTitle(Resource.String.MainSyncConfirmationTitle)
                       .SetIcon(Android.Resource.Drawable.IcDialogAlert)
                       .SetCancelable(true)
                       .SetPositiveButton(Android.Resource.String.Ok,
                         (s, a) =>
                         {
                             progressDialog = ProgressDialog.Show(this, GetString(Resource.String.MainSyncProgressTitle),
                             GetString(Resource.String.MainSyncProgressMessage), true, false);
                             ThreadPool.QueueUserWorkItem((o) => DoSynchronisation());
                         })
                       .SetNegativeButton(Android.Resource.String.Cancel,
                         (IDialogInterfaceOnClickListener)null);
                return builder.Create();
            }
            return null;
        }

        public override void OnBackPressed()
        {
            Toast.MakeText(this, Resource.String.MainBackButtonPressed, ToastLength.Short).Show();
            base.OnBackPressed();
        }

        private void DoSynchronisation()
        {
            try
            {
                //First we'll locate new entries and send them to the server
                var helper = ((TimeSheetApplication)Application).DatabaseHelper;
                var cursor = helper.FetchUpdatedEntries(prefs.UserID);
                if (cursor.Count > 0)
                {
                    StringBuilder builder = new StringBuilder();
                    //builder.AppendLine("<Entries>");
                    cursor.MoveToFirst();
                    while (!cursor.IsAfterLast)
                    {
                        Entry entry = Entry.FromCursor(cursor);
#if NO_SERVER_HACK
                        //This entry is going to be *hackety-hack* stored locally as we don't really have a server
                        //To cater for the server absence we should mark the entry as synced now
                        entry.Synced = true;
#endif
                        builder.AppendLine(entry.ToXMLString());
                        cursor.MoveToNext();
                    }
                    //builder.AppendLine("</Entries>");
                    Log.Info(Consts.Tag, builder.ToString());

                    //Send this XML string to the server
                    Common.CallWebServiceToSendUserUpdatedEntries(this, prefs.UserEmail, prefs.UserPassword, builder.ToString());
                    RunOnUiThread(() => Common.LongToast(Resource.String.MainSyncUpSuccessful));
                    //Now update the local copies to show they are synced
                    helper.UpdateEntriesToBeSynced(prefs.UserID);
                }
                else
                    RunOnUiThread(() => Common.LongToast(Resource.String.MainSyncUpUnnecessary));

                //Now we'll ask the server for its updates and update the local database
                string loginXml = Common.CallWebServiceToLoginAndGetUserData(this, prefs.UserEmail, prefs.UserPassword) ?? "";
                if (loginXml != "")
                {
                    try
                    {
                        Log.Info(Consts.Tag, "Processing user data for sync");
                        Common.ProcessUserData(this, loginXml, false, null, null);
                    }
                    catch (Exception e)
                    {
                        RunOnUiThread(() => Common.ShortToast(e.Message));
                    }                    
                }
                else
                    RunOnUiThread(() => Common.LongToast(Resource.String.MainSyncDownFailed));

            }
            catch (Java.Lang.Throwable t)
            {
                Common.LogException(t);
                RunOnUiThread(() => Common.LongToast(Resource.String.MainSyncFailed));
            }
            RunOnUiThread(() => progressDialog.Dismiss());
        }
    }
}

