using System.Collections.Generic;

using Android.Content;

using Android.Database;
using Android.Database.Sqlite;
using Android.Util;

//Notes
/* To examine the database from a workstation, use sqlite3 in an adb shell against the app on an emulator
 * For example:
 * 
 * adb -e shell
 * sqlite3 /data/data/com.blong.TimeSheet/databases/TimeSheets.db
 * .tables
 * .schema Entries
 * .dump Entries
 * 
 * .quit
 * exit
 */


namespace TimeSheet
{
    class DBHelper
    {
        private Context context;
        private SQLHelper helper;
        private SQLiteDatabase db;

        public DBHelper(Context context)
        {
            this.context = context;
            helper = new SQLHelper(context);
            Open();
        }

        public DBHelper Open()
        {
            if (db == null)
                db = helper.WritableDatabase;
            return this;
        }

        public void Close()
        {
            helper.Close();
        }

        #region User-related DB routines
        public void ResetUser(User user)
        {
            ContentValues content = new ContentValues();
            content.Put("_id", user.ID);
            content.Put("name", user.Name);
            ICursor cursor = db.Query("Users", new string[] { "_id" }, "_id=?", new string[] { user.ID.ToString() }, null, null, null);
            if (cursor.Count == 0)
                db.Insert("Users", null, content);
            else
                db.Update("Users", content, "_id=?", new string[] { user.ID.ToString() });
        }
        #endregion

        #region Client-related DB routines
        public void ResetClients(List<Client> clients)
        {
            db.BeginTransaction();
            try
            {
                //Delete current clients from database
                db.Delete("Clients", "", null);
                //Add clients in list to database
                ContentValues content = new ContentValues();
                foreach (Client client in clients)
                {
                    content.Put("_id", client.ID);
                    content.Put("name", client.Name);
                    content.Put("defaultRate", client.DefaultHourlyRate);
                    content.Put("comment", client.Comment);
                    db.Insert("Clients", null, content);
                }
                db.SetTransactionSuccessful();
            }
            finally
            {
                db.EndTransaction();
            }
        }

        public Client FetchClient(int id)
        {
            ICursor c = db.Query("Clients",
                new string[] { "_id", "name", "defaultRate", "comment" },
                "_id=?", new string[] { id.ToString() }, null, null, null);
            c.MoveToFirst();
            return Client.FromCursor(c);
        }

        public ICursor FetchAllClientsCursor()
        {
            return db.Query("Clients",
                new string[] { "_id", "name", "defaultRate", "comment" },
                null, null, null, null, "name");
        }
        #endregion

        #region Entry-related DB routines
        public void ResetEntries(List<Entry> entries)
        {
            db.BeginTransaction();
            try
            {
                TimeSheetPreferences prefs = new TimeSheetPreferences(context.ApplicationContext);
                //Delete current clients from database
                db.Delete("Entries", "userId=? and synced=1", new string[] { prefs.UserID.ToString() });
                //Add clients in list to database
                ContentValues content = new ContentValues();
                foreach (Entry entry in entries)
                {
                    content.Put("userId", entry.UserID);
                    content.Put("clientId", entry.ClientID);
                    content.Put("startTime", entry.StartTime.ToString(Consts.DateTimeFmtSQL));
                    content.Put("endTime", entry.EndTime.ToString(Consts.DateTimeFmtSQL));
                    content.Put("rate", entry.HourlyRate);
                    content.Put("synced", true);
                    content.Put("comment", entry.Comment);
                    db.Insert("Entries", null, content);
                }
                db.SetTransactionSuccessful();
            }
            finally
            {
                db.EndTransaction();
            }
        }

        public ICursor FetchUpdatedEntries(int userId)
        {
            return db.Query("Entries", null, "userId=? and synced=0", new string[] { userId.ToString() }, null, null, null);
        }

        public void UpdateEntriesToBeSynced(int userId)
        {
            ContentValues content = new ContentValues();
            content.Put("synced", true);
            ICursor cursor;
            cursor = db.Query("Entries", new string[] { "_id and synced=0" }, "_id=?", new string[] { userId.ToString() }, null, null, null);
            Log.Info(Consts.Tag, "Before local update, unsynced entries = " + cursor.Count.ToString());
            cursor = db.Query("Entries", new string[] { "_id and synced=1" }, "_id=?", new string[] { userId.ToString() }, null, null, null);
            Log.Info(Consts.Tag, "Before local update, synced entries = " + cursor.Count.ToString());
            db.Update("Entries", content, "userId=? and synced=0", new string[] { userId.ToString() });
            cursor = db.Query("Entries", new string[] { "_id and synced=0" }, "_id=?", new string[] { userId.ToString() }, null, null, null);
            Log.Info(Consts.Tag, "After local update, unsynced entries = " + cursor.Count.ToString());
            cursor = db.Query("Entries", new string[] { "_id and synced=1" }, "_id=?", new string[] { userId.ToString() }, null, null, null);
            Log.Info(Consts.Tag, "After local update, synced entries = " + cursor.Count.ToString());
        }

        public void AddTimeSheetEntry(Entry entry)
        {
            ContentValues content = new ContentValues();
            content.Put("userId", entry.UserID);
            content.Put("clientId", entry.ClientID);
            //When using the default approach, the time, when after midday is going in as 12 hour clock time, thus a.m.
            content.Put("startTime", entry.StartTime.ToString(Consts.DateTimeFmtSQL));
            content.Put("endTime", entry.EndTime.ToString(Consts.DateTimeFmtSQL));
            content.Put("rate", entry.HourlyRate);
            content.Put("synced", entry.Synced);
            content.Put("comment", entry.Comment);
            entry.ID = (int)db.Insert("Entries", null, content);
        }
        #endregion

        public ICursor GenerateReportList(int userId)
        {
            const string qry =
                "select c._id, " +
                "c.name || '\n' || " +
                    "cast(strftime('%d/%m/%Y', e.startTime) as text) || ' ' || " +
                    "cast(strftime('%H:%M', e.startTime) as text) || '-' || " +
                    "cast(strftime('%H:%M', e.endTime) as text) || " +
                    "'\n' || e.rate || '/hr' as desc, " +
                    "e.synced, e.comment " +
                "from Entries e inner join Clients c " +
                "where e.userId=? and e.clientId=c._id " +
                "order by e.startTime";
            return db.RawQuery(qry, new string[] { userId.ToString() });
        }

        #region Nested SQLiteOpenHelper decendant class
        private class SQLHelper : SQLiteOpenHelper
        {
            public const int DB_VER = 3;
            private const string DB_NAME = "TimeSheets.db";

            public SQLHelper(Context context) :
                base(context, DB_NAME, null, DB_VER)
            {
            }

            public override void OnCreate(SQLiteDatabase db)
            {
                const string DB_CREATE_USERS =
                    "create table Users (" +
                    "_id integer primary key, " +
                    "name text not null" +
                    ")";
                const string DB_CREATE_ENTRIES =
                    "create table Entries (" +
                    "_id integer primary key autoincrement, " +
                    "userId integer not null, " +
                    "clientId integer not null, " +
                    "startTime datetime not null, " +
                    "endTime datetime not null, " +
                    "rate real not null, " +
                    "synced int not null, " +
                    "comment text" +
                    ")";
                const string DB_CREATE_CLIENTS =
                    "create table Clients (" +
                    "_id integer primary key, " +
                    "name text not null, " +
                    "defaultRate real not null, " +
                    "comment text" +
                    ")";
                db.ExecSQL(DB_CREATE_USERS);
                db.ExecSQL(DB_CREATE_ENTRIES);
                db.ExecSQL(DB_CREATE_CLIENTS);
            }

            public override void OnUpgrade(SQLiteDatabase db, int oldVersion, int newVersion)
            {
                Log.Warn(Consts.Tag, "Upgrading database from version " + oldVersion +
                    " to " + newVersion + ", which will destroy all old data");
                DropDB(db);
                OnCreate(db);
            }

            private void DropDB(SQLiteDatabase db)
            {
                const string DB_DESTROY_USERS = "drop table if exists Users";
                const string DB_DESTROY_ENTRIES = "drop table if exists Entries";
                const string DB_DESTROY_CLIENTS = "drop table if exists Clients";
                db.ExecSQL(DB_DESTROY_USERS);
                db.ExecSQL(DB_DESTROY_ENTRIES);
                db.ExecSQL(DB_DESTROY_CLIENTS);
            }
        }
        #endregion
    }
}