using System;
using System.Text;

using Android.OS;

using System.Xml;
using Android.Util;
using Android.Database;
using System.Globalization;

namespace TimeSheet
{
    public abstract class DBBaseClass<T> where T : class
    {
        public static T FromXML(XmlNode node) { return null; }
        public abstract string ToXMLString();
        public static T FromBundle(Bundle b) { return null; }
        public abstract Bundle ToBundle();
    }

    #region User class
    public class User : DBBaseClass<User>
    {
        public int ID { get; set; }
        public string Name { get; set; }

        public User()
        {
            ID = -1;
        }

        public new static User FromXML(XmlNode node)
        {
            if (node == null)
                return null;
            User user = new User();
            user.ID = Common.XMLNodeInteger(node, "ID");
            user.Name = Common.XMLNodeString(node, "Name");
            return user;
        }

        public override string ToXMLString()
        {
            StringBuilder xml = new StringBuilder();
            xml.Append("<User>");
            if (ID != -1)
            {
                xml.AppendFormat("<ID>{0}</ID>", ID);
                if ((Name ?? "") != "")
                    xml.AppendFormat("<Name>{0}</Name>", Name);
            }
            xml.Append("</User>");
            return xml.ToString();
        }

        public new static User FromBundle(Bundle b)
        {
            if (b == null)
                return null;
            User user = new User();
            user.ID = b.GetInt("id");
            user.Name = b.GetString("name");
            return user;
        }

        public override Bundle ToBundle()
        {
            Bundle b = new Bundle();
            b.PutInt("id", ID);
            b.PutString("name", Name);
            return b;
        }
    }
    #endregion

    #region Entry class
    public class Entry : DBBaseClass<Entry>
    {
        public int ID { get; set; }
        public int UserID { get; set; }
        public int ClientID { get; set; }
        public DateTime StartTime { get; set; }
        public DateTime EndTime { get; set; }
        public double HourlyRate { get; set; }
        public bool Synced { get; set; }
        public string Comment { get; set; }

        public Entry()
        {
            ID = -1;
            UserID = -1;
            ClientID = -1;
        }

        public new static Entry FromXML(XmlNode node)
        {
            if (node == null)
                return null;
            Entry entry = new Entry();
            entry.UserID = Common.XMLNodeInteger(node, "UserID");
            entry.ClientID = Common.XMLNodeInteger(node, "ClientID");
            entry.StartTime = Common.XMLNodeDateTime(node, "StartTime");
            entry.EndTime = Common.XMLNodeDateTime(node, "EndTime");
            entry.HourlyRate = Common.XMLNodeDouble(node, "HourlyRate");
            entry.Comment = Common.XMLNodeString(node, "Comment");
            entry.Synced = Common.XMLNodeBoolean(node, "Synced");
            return entry;
        }

        public override string ToXMLString()
        {
            StringBuilder xml = new StringBuilder();
            xml.Append("<Entry>");
            if ((ID) != -1)
                xml.AppendFormat("<ID>{0}</ID>", ID);
            if ((UserID) != -1)
                xml.AppendFormat("<UserID>{0}</UserID>", UserID);
            if ((ClientID) != -1)
                xml.AppendFormat("<ClientID>{0}</ClientID>", ClientID);
            xml.AppendFormat("<StartTime>{0}</StartTime>", StartTime.ToString(Consts.DateTimeFmtXML));
            xml.AppendFormat("<EndTime>{0}</EndTime>", EndTime.ToString(Consts.DateTimeFmtXML));
            xml.AppendFormat("<HourlyRate>{0}</HourlyRate>", HourlyRate);
            xml.AppendFormat("<Synced>{0}</Synced>", Synced);
            if ((Comment ?? "") != "")
                xml.AppendFormat("<Comment>{0}</Comment>", Comment);
            xml.Append("</Entry>");
            return xml.ToString();
        }

        public new static Entry FromBundle(Bundle b)
        {
            if (b == null)
                return null;
            Entry entry = new Entry();
            entry.ID = b.GetInt("id");
            entry.UserID = b.GetInt("userid");
            entry.ClientID = b.GetInt("clientid");
            entry.StartTime = DateTime.FromOADate(b.GetDouble("starttime"));
            entry.EndTime = DateTime.FromOADate(b.GetDouble("endtime"));
            entry.HourlyRate = b.GetDouble("hourlyrate");
            entry.Synced = b.GetBoolean("synced");
            entry.Comment = b.GetString("comment");
            return entry;
        }

        public override Bundle ToBundle()
        {
            Bundle b = new Bundle();
            b.PutInt("id", ID);
            b.PutInt("userid", UserID);
            b.PutInt("clientid", ClientID);
            b.PutDouble("starttime", StartTime.ToOADate());
            b.PutDouble("endtime", EndTime.ToOADate());
            b.PutDouble("hourlyrate", HourlyRate);
            b.PutBoolean("synced", Synced);
            b.PutString("comment", Comment);
            return b;
        }

        public static Entry FromCursor(ICursor c)
        {
            Log.Info(Consts.Tag, "Entry.FromCursor()");
            if ((c == null) || c.IsBeforeFirst || c.IsAfterLast)
            {
                Log.Warn(Consts.Tag, "Client.FromCursor() returned prematurely");
                return null;
            }
            Entry entry = new Entry();
            entry.ID = c.GetInt(c.GetColumnIndex("_id"));
            entry.UserID = c.GetInt(c.GetColumnIndex("userId"));
            entry.ClientID = c.GetInt(c.GetColumnIndex("clientId"));
            Log.Info(Consts.Tag, c.GetString(c.GetColumnIndex("startTime")));
            entry.StartTime = DateTime.ParseExact(c.GetString(c.GetColumnIndex("startTime")),
                Consts.DateTimeFmtSQL, CultureInfo.InvariantCulture);
            Log.Info(Consts.Tag, c.GetString(c.GetColumnIndex("endTime")));
            entry.EndTime = DateTime.ParseExact(c.GetString(c.GetColumnIndex("endTime")),
                Consts.DateTimeFmtSQL, CultureInfo.InvariantCulture);
            entry.HourlyRate = c.GetDouble(c.GetColumnIndex("rate"));
            entry.Synced = c.GetInt(c.GetColumnIndex("synced")) != 0;
            entry.Comment = c.GetString(c.GetColumnIndex("comment"));
            return entry;
        }
    }
    #endregion

    #region Client class
    public class Client : DBBaseClass<Client>
    {
        public int ID { get; set; }
        public string Name { get; set; }
        public double DefaultHourlyRate { get; set; }
        public string Comment { get; set; }

        public Client()
        {
            ID = -1;
        }

        public new static Client FromXML(XmlNode node)
        {
            if (node == null)
                return null;
            Client client = new Client();
            client.ID = Common.XMLNodeInteger(node, "ID");
            client.Name = Common.XMLNodeString(node, "Name");
            client.DefaultHourlyRate = Common.XMLNodeDouble(node, "DefaultHourlyRate");
            client.Comment = Common.XMLNodeString(node, "Comment");
            return client;
        }

        public override string ToXMLString()
        {
            StringBuilder xml = new StringBuilder();
            xml.Append("<Client>");
            if (ID != -1)
            {
                xml.AppendFormat("<ID>{0}</ID>", ID);
                if ((Name ?? "") != "")
                    xml.AppendFormat("<Name>{0}</Name>", Name);
                xml.AppendFormat("<DefaultHourlyRate>{0}</DefaultHourlyRate>", DefaultHourlyRate);
                if ((Comment ?? "") != "")
                    xml.AppendFormat("<Comment>{0}</Comment>", Comment);
            }
            xml.Append("</Client>");
            return xml.ToString();
        }

        public new static Client FromBundle(Bundle b)
        {
            if (b == null)
                return null;
            Client client = new Client();
            client.ID = b.GetInt("id");
            client.Name = b.GetString("name");
            client.DefaultHourlyRate = b.GetDouble("defaulthourlyrate");
            client.Comment = b.GetString("comment");
            return client;
        }

        public override Bundle ToBundle()
        {
            Bundle b = new Bundle();
            b.PutInt("id", ID);
            b.PutString("name", Name);
            b.PutDouble("defaulthourlyrate", DefaultHourlyRate);
            b.PutString("comment", Comment);
            return b;
        }

        public static Client FromCursor(ICursor c)
        {
            if ((c == null) || c.IsBeforeFirst || c.IsAfterLast)
            {
                Log.Warn(Consts.Tag, "Client.FromCursor() returned prematurely");
                return null;
            }
            Client client = new Client();
            client.ID = c.GetInt(c.GetColumnIndex("_id"));
            client.Name = c.GetString(c.GetColumnIndex("name"));
            client.DefaultHourlyRate = c.GetDouble(c.GetColumnIndex("defaultRate"));
            client.Comment = c.GetString(c.GetColumnIndex("comment"));
            return client;
        }
    }
    #endregion
}